require 'scripts/core/core.lua'
--------------------------------------------------------------------------------
-- SettingsPage Object script
--------------------------------------------------------------------------------
SettingsPage = class()

-- constants
SETTINGS_STATE_IDLE = 0
SETTINGS_STATE_ANIM = 1
SETTINGS_STATE_NOT_ACTIVE = 2
SETTINGS_STATE_SCROLLING = 3

FAST_SPEED_RATIO = 2.5

-- Callback when object is added to the world
function SettingsPage:start()

	self.arrow_left = ScriptComponent_getScriptTable(self.arrow_left)
	self.arrow_right = ScriptComponent_getScriptTable(self.arrow_right)
	self.switch_hand_button = ScriptComponent_getScriptTable(self.switch_hand_button)
	self.validate_button = ScriptComponent_getScriptTable(self.validate_button)
	self.bilinear_button = ScriptComponent_getScriptTable(self.bilinear_button)
	--self.layout_validate_button = ScriptComponent_getScriptTable(self.layout_validate_button)
	
	self.group_count = VC_getLayoutGroupCount()
	
	self.can_cycle = self.group_count > 2
	
	self.group_current_alt = {}
	for i = 1, VC_getLayoutGroupCount() do
		self.group_current_alt[i] = 1
	end
	
	self.is_active = true
	
	self.outlightH_node = Component_getNode(self.outlightH)
	self.outlightH_animator = WorldNode_getComponentByTypeName(self.outlightH_node, "animator")
	self.outlightV_node = Component_getNode(self.outlightV)
	self.outlightV_animator = WorldNode_getComponentByTypeName(self.outlightV_node, "animator")
	
	-- Bilinear switch button : Adapt clickable area size to the text size
	local labelWidth, labelHeight = VisualComponent_getSize(self.label_bilinear_guide_message)
	local clickWidth, clickHeight = ClickableComponent_getBoxShapeSize(self.bilinear_clickable_area)
	
	ClickableComponent_setBoxShape(self.bilinear_clickable_area, labelWidth + 20, math.max(clickHeight, labelHeight))
	ClickableComponent_setCenter(self.bilinear_clickable_area, -(labelWidth / 2) + 30, -30)
	
	local offPosX, offPosY = WorldNode_getLocalPosition(self.label_bilinear_off_node)
	
	local labelOff = WorldNode_getComponentByTypeName(self.label_bilinear_off_node, "label")
	local labelBar = WorldNode_getComponentByTypeName(self.label_bilinear_bar_node, "label")
	
	local labelOffWidth, labelOffHeight = VisualComponent_getSize(labelOff)
	local labelBarWidth, labelBarHeight = VisualComponent_getSize(labelBar)
	
	WorldNode_setLocalPosition(self.label_bilinear_bar_node, offPosX - labelOffWidth - 5, offPosY)
	WorldNode_setLocalPosition(self.label_bilinear_on_node, offPosX - labelOffWidth - 5 - labelBarWidth - 7, offPosY)
	
end

-- Callback when object is removed from the world
function SettingsPage:stop()
end

function SettingsPage:setupLayouts(layout_pres_prefab)
	local layout_node = nil
	
	self.layout_nodes = {}
	
	layout_node = Scene_instantiate(layout_pres_prefab)
	WorldNode_setName(layout_node, "left_layout_2")
	WorldNode_addChildNode(self.layouts_node, layout_node)
	layout_node = WorldNode_getChildByName(layout_node, "button")
	self.layout_nodes[1] = layout_node
	Component_disable(WorldNode_getScriptComponent(layout_node, "GUIButton").clickable)
	
	layout_node = Scene_instantiate(layout_pres_prefab)
	WorldNode_setName(layout_node, "left_layout_1")
	WorldNode_addChildNode(self.layouts_node, layout_node)
	layout_node = WorldNode_getChildByName(layout_node, "button")
	self.layout_nodes[2] = layout_node
	Component_disable(WorldNode_getScriptComponent(layout_node, "GUIButton").clickable)
	
	layout_node = Scene_instantiate(layout_pres_prefab)
	WorldNode_setName(layout_node, "active_layout")
	WorldNode_addChildNode(self.layouts_node, layout_node)
	layout_node = WorldNode_getChildByName(layout_node, "button")
	self.layout_nodes[3] = layout_node

	self.layout_validate_button = WorldNode_getScriptComponent(layout_node, "GUIButton")
	self.layout_validate_button.listener = self
	Component_enable(self.layout_validate_button._ptr)

	layout_node = Scene_instantiate(layout_pres_prefab)
	WorldNode_setName(layout_node, "active_layout_alt")
	WorldNode_addChildNode(self.layouts_node, layout_node)
	layout_node = WorldNode_getChildByName(layout_node, "button")
	self.layout_node_alt = layout_node
	WorldNode_setEnabled(self.layout_node_alt, false)
	Component_disable(WorldNode_getScriptComponent(layout_node, "GUIButton").clickable)

	layout_node = Scene_instantiate(layout_pres_prefab)
	WorldNode_setName(layout_node, "right_layout_1")
	WorldNode_addChildNode(self.layouts_node, layout_node)
	layout_node = WorldNode_getChildByName(layout_node, "button")
	self.layout_nodes[4] = layout_node
	Component_disable(WorldNode_getScriptComponent(layout_node, "GUIButton").clickable)
	
	layout_node = Scene_instantiate(layout_pres_prefab)
	WorldNode_setName(layout_node, "right_layout_2")
	WorldNode_addChildNode(self.layouts_node, layout_node)
	layout_node = WorldNode_getChildByName(layout_node, "button")
	self.layout_nodes[5] = layout_node
	Component_disable(WorldNode_getScriptComponent(layout_node, "GUIButton").clickable)

	self:resolveAnimators()
end

function SettingsPage:setLayers(ascending)
	
	local startLayer = ascending and 4 or 28
	local layerRange = ascending and 6 or -6
	
	for i = 1, 5 do
		self:setItemLayers(WorldNode_getScriptComponent(self.layout_nodes[i], "GUIButton"), startLayer + (i - 1) * layerRange)
	end
	
end

function SettingsPage:setItemLayers(item, baseLayer)
	
	VisualComponent_setLayer(item.tv_icon,           baseLayer)
	VisualComponent_setLayer(item.tv_background,     baseLayer + 1)
	VisualComponent_setLayer(item.tv_screen_bottom,  baseLayer + 2)
	VisualComponent_setLayer(item.tv_screen_top,     baseLayer + 2)
	VisualComponent_setLayer(item.drc_icon,          baseLayer + 3)
	VisualComponent_setLayer(item.drc_background,    baseLayer + 4)
	VisualComponent_setLayer(item.drc_screen_bottom, baseLayer + 5)
	VisualComponent_setLayer(item.drc_screen_top,    baseLayer + 5)
	
end

function SettingsPage:resolveAnimators()
	AnimatorComponent_resolveBindings(self.switch_animator_top)
	AnimatorComponent_resolveBindings(self.switch_animator_bottom)
	AnimatorComponent_resolveBindings(self.move_right_animator)
	AnimatorComponent_resolveBindings(self.move_left_animator)
end

-- Initalize layout menu on entering
function SettingsPage:enter()
	
	self.currentLayout = VC_getCurrentConfiguration()
	
	local alt
	self.current_layout_group, alt = getGroupAndAltFromLinearIndex(self.currentLayout)
	self.group_current_alt[self.current_layout_group] = alt
	
	self.state = SETTINGS_STATE_NOT_ACTIVE
	self.animation_controller = nil
	
	
	Component_disable(self.label_layout_name_alt)
	Component_disable(self.label_layout_desc_alt)
	
	LabelComponent_setTextFromStringId(self.label_layout_name, VC_getScreenConfigurationName(self.currentLayout))
	LabelComponent_setTextFromStringId(self.label_layout_desc, VC_getScreenConfigurationDesc(self.currentLayout))

	self:computeLayoutsPresentation()
	AnimatorComponent_setCurrentTime(self.move_left_animator,0.0)
	AnimatorComponent_setCurrentTime(self.move_right_animator,0.0)
	
	VisualComponent_setAlpha(self.label_layout_name, 1.0)
	VisualComponent_setAlpha(self.label_layout_desc, 1.0)
	
	-- local drcRotation = VC_getScreenDrcRotation(self.currentLayout)
			
	-- -- choose correct outlight sprite
	-- if drcRotation == 90 or drcRotation == 270 then
	-- 	VisualComponent_setAlpha(self.outlightV, 1.0)
	-- 	VisualComponent_setAlpha(self.outlightH, 0.0)
	-- else
	-- 	VisualComponent_setAlpha(self.outlightV, 0.0)
	-- 	VisualComponent_setAlpha(self.outlightH, 1.0)
	-- end
	self:selectOutlight()
	
	-- scroll arrows
	self:showHideArrows()
	
	-- Choose correct bilinear mode
	self.bilinear = VC_getBilinearMode()
	self.originalBilinear = self.bilinear

	local animation = nil

	if self.bilinear then
		animation = self.bilinear_on_anim
	else
		animation = self.bilinear_off_anim
	end
	
	Component_enable(animation)
	AnimatorComponent_setCurrentTime(animation, 0.3)
	

	g_menu:applyScreensFilteringMode()
	
	local isBilinearEnabled = not isBilinearDisabledForLayout(self.currentLayout)
	self:enableBilinearOption(isBilinearEnabled, true)
	
	self:enableClickables(false)
end

function SettingsPage:switchBilinearMode()
	self.bilinear = not self.bilinear
	
	--VC_setBilinearMode(self.bilinear)

	g_menu:applyScreensFilteringMode()

	return self.bilinear
end



-- Callback called every frames
function SettingsPage:update(dt)
	
	if not self.is_active then
		return
	end

	local left_pressed = GUI:isLeftPressed()
	local right_pressed = GUI:isRightPressed()
	local up_pressed = GUI:isUpPressed()
	local down_pressed = GUI:isDownPressed()
	local alt_pressed = up_pressed or down_pressed
	
	if self.state == SETTINGS_STATE_NOT_ACTIVE then
		
			self.state = SETTINGS_STATE_IDLE
			self:enableClickables(true)

	elseif self.state == SETTINGS_STATE_IDLE then
		
		if GUI.pressedCommandTarget == nil and GUI.dragCommandTarget == nil then -- disable keys when a touch button is pressed
		
			local back_pressed = GUI:isCancelPressed()
			local validate_pressed = GUI:isValidatePressed()
			
			if left_pressed and (self.can_cycle or self.current_layout_group > 1) then
				Component_enable(self.label_layout_name_alt)
				Component_enable(self.label_layout_desc_alt)
				self:chooseLeftLayout()
				self.arrow_left:playClickedAnimation()

			elseif right_pressed and (self.can_cycle or self.current_layout_group < self.group_count) then
				Component_enable(self.label_layout_name_alt)
				Component_enable(self.label_layout_desc_alt)
				self:chooseRightLayout()
				self.arrow_right:playClickedAnimation()

			elseif alt_pressed and VC_getLayoutGroupConfigCount(self.current_layout_group) > 1 then
				Component_enable(self.label_layout_name_alt)
				Component_enable(self.label_layout_desc_alt)
				
				if up_pressed then
					self:chooseAltLayout(self.switch_animator_top)
				else
					self:chooseAltLayout(self.switch_animator_bottom)
				end
				
				self.switch_hand_button:playClickedAnimation()
				
			elseif back_pressed or validate_pressed then
				
				Input_freeze()
				
				VC_playSound(g_menu.sound_select, 1.0)

				if validate_pressed then
					self.validate_button:playClickedAnimation()
					self.layout_validate_button:playClickedAnimation()
					self:applyAndExit()
				else
					self:cancelAndExit()
				end
				
			elseif self.isBilinearOptionVisible and GUI:isKeyPressed(PAD_DEVICE, GAMEPAD_RIGHT_STICK) then
				
				local isActivated = self:switchBilinearMode()
				local animation = nil
				
				if isActivated then
					animation = self.bilinear_on_anim
				else
					
					animation = self.bilinear_off_anim
				end
				
				Component_enable(animation)
				AnimatorComponent_reset(animation)
				AnimatorComponent_replay(animation)
				
				self.bilinear_button:playClickedAnimation()
				VC_playSound(g_menu.sound_check, 1.0)
				
			end
		end
		
	elseif self.state == SETTINGS_STATE_ANIM then
		
		if left_pressed or right_pressed then
			local speed = AnimatorComponent_getSpeed(self.animation_controller)
			AnimatorComponent_setSpeed(self.animation_controller, speed * FAST_SPEED_RATIO)
		end
		
		-- On Animation End
		if AnimatorComponent_isStopped(self.animation_controller) then
			-- AnimatorComponent_setCurrentTime(self.animation_controller,0.0)
			-- AnimatorComponent_stop(self.animation_controller)
			AnimatorComponent_reset(self.animation_controller)
			AnimatorComponent_setCurrentTime(self.animation_controller,0.0)
			AnimatorComponent_stop(self.animation_controller)
			
			VisualComponent_setAlpha(self.label_layout_name, 1.0)
			VisualComponent_setAlpha(self.label_layout_desc, 1.0)
			
			Component_disable(self.animation_controller)
			Component_disable(self.label_layout_name_alt)
			Component_disable(self.label_layout_desc_alt)
			WorldNode_setEnabled(self.layout_node_alt, false)
			self:computeLayoutsPresentation()
			LabelComponent_setTextFromStringId(self.label_layout_name, VC_getScreenConfigurationName(self.currentLayout))
			LabelComponent_setTextFromStringId(self.label_layout_desc, VC_getScreenConfigurationDesc(self.currentLayout))
			self.state = SETTINGS_STATE_IDLE
			
			self:showHideArrows()
			
			self:enableClickables(true)
			
			-- enable blinking animation
			Component_enable(self.outlightH_animator)
			AnimatorComponent_reset(self.outlightH_animator)
			Component_enable(self.outlightV_animator)
			AnimatorComponent_reset(self.outlightV_animator)
			
		end
	end
end

-- Callback when object is a button and is pressed
function SettingsPage:onButtonClick(button)
	
	if not self.is_active then
		return
	end
	
	if self.state == SETTINGS_STATE_IDLE then
		if button == self.arrow_left then
			Component_enable(self.label_layout_name_alt)
			Component_enable(self.label_layout_desc_alt)
			self:chooseLeftLayout()
		elseif button == self.arrow_right then
			Component_enable(self.label_layout_name_alt)
			Component_enable(self.label_layout_desc_alt)
			self:chooseRightLayout()
		elseif button == self.switch_hand_button then
			Component_enable(self.label_layout_name_alt)
			Component_enable(self.label_layout_desc_alt)
			
			--self:chooseAltLayout(self.switch_animator_top)
			self:chooseAltLayout(self.switch_animator_bottom)
			
			-- enable this if you want the animation to alternate
			-- if self.group_current_alt[self.current_layout_group] == 1 then
			-- 	self:chooseAltLayout(self.switch_animator_top)
			-- elseif self.group_current_alt[self.current_layout_group] == 2 then
			-- 	self:chooseAltLayout(self.switch_animator_bottom)
			-- end
		elseif self.isBilinearOptionVisible and button == self.bilinear_button then
			
			local isActivated = self:switchBilinearMode()
			local animation = nil
			
			if isActivated then
				animation = self.bilinear_on_anim
			else
				
				animation = self.bilinear_off_anim
			end
			
			Component_enable(animation)
			AnimatorComponent_reset(animation)
			AnimatorComponent_replay(animation)
			
			
			VC_playSound(g_menu.sound_check, 1.0)
		
		elseif button == self.validate_button or button == self.layout_validate_button then
		
			self:applyAndExit()
			
		end
	else
		local speed = AnimatorComponent_getSpeed(self.animation_controller)
		AnimatorComponent_setSpeed(self.animation_controller, speed * FAST_SPEED_RATIO)
	end
end

function SettingsPage:onButtonPress(button)

	if self.is_active then
		if button ~= self.bilinear_button then
			VC_playSound(g_menu.sound_select, 1.0)
		end
	end
end

function SettingsPage:onAnimationEvent(eventId, eventTime, triggerTime)
	
	if eventId == "middle" then
		self:displaySwitchHandIcon()
		
		self:selectOutlight()
		
		local isBilinearEnabled  = not isBilinearDisabledForLayout(self.currentLayout)
		self:enableBilinearOption(isBilinearEnabled)
	end
end

function SettingsPage:applyAndExit()
	
	VC_setCurrentConfiguration(self.currentLayout)
	
	g_menu:applyBackgrounds()

	g_menu:computeBackgroundScreensLayout()
	
	VC_playSound(g_menu.sound_popup_ok, 1.0)
	
	VC_setBilinearMode(self.bilinear)
	
	g_menu:switchPage(MAIN_PAGE)
	
	GUI:focusElement(nil)
	
	self:enableClickables(false)
end

function SettingsPage:cancelAndExit()
	VC_playSound(g_menu.sound_popup_cancel, 1.0)
	g_menu:switchPage(MAIN_PAGE)
	
	GUI:focusElement(nil)
	
	self:enableClickables(false)
end



-- show/hide arrows according to cycle possibility
function SettingsPage:showHideArrows()
	
	if not self.can_cycle and self.current_layout_group == self.group_count then
		WorldNode_setEnabled(self.arrow_right_node, false)
	else
		WorldNode_setEnabled(self.arrow_right_node, true)
	end
	
	if not self.can_cycle and self.current_layout_group == 1 then
		WorldNode_setEnabled(self.arrow_left_node, false)
	else
		WorldNode_setEnabled(self.arrow_left_node, true)
	end
	
end

function SettingsPage:enableClickables(enable, enableArrows, enableScroller)
	
	if enableArrows == nil then
		enableArrows = enable
	end
	
	if enableScroller == nil then
		enableScroller = enable
	end
	
	Component_setEnabled(self.arrow_left.clickable, enableArrows)
	Component_setEnabled(self.arrow_right.clickable, enableArrows)
	Component_setEnabled(self.switch_hand_button.clickable, enable)
	Component_setEnabled(self.validate_button.clickable, enable)
	if self.isBilinearOptionVisible then
		Component_setEnabled(self.bilinear_button.clickable, enable)
	else
		Component_setEnabled(self.bilinear_button.clickable, false)
	end
		
	Component_setEnabled(self.layout_validate_button.clickable, enable)
	Component_setEnabled(self.layouts_scroller_clickable, enableScroller)
end

function SettingsPage:launchAnimation(animator, keepAnimState)
	
	self.animation_controller = animator
	
	local isMovingLeftOrRight = animator == self.move_left_animator or animator == self.move_right_animator
	self:enableClickables(false, true, isMovingLeftOrRight)
	
	-- disable blinking animation
	Component_disable(self.outlightH_animator)
	Component_disable(self.outlightV_animator)
	
	Component_enable(self.animation_controller)

	if keepAnimState then
		AnimatorComponent_play(self.animation_controller)
	else
		AnimatorComponent_setSpeed(self.animation_controller, 1.0)
		AnimatorComponent_replay(self.animation_controller)
	end
				
	self.state = SETTINGS_STATE_ANIM
	
end


-- Callback when object is a button and is released
function SettingsPage:onReleased()

end

function SettingsPage:selectOutlight(layout)
	
	if layout == nil then
		layout = self.currentLayout
	end
	
	local drcRotationForNextLayout = VC_getScreenDrcRotation(layout)
	
	if drcRotationForNextLayout == 90 or drcRotationForNextLayout == 270 then
		-- Component_enable(self.outlightV)
		-- Component_disable(self.outlightH)
		WorldNode_setEnabled(self.outlightV_node, true)
		WorldNode_setEnabled(self.outlightH_node, false)
	else
		-- Component_disable(self.outlightV)
		-- Component_enable(self.outlightH)
		WorldNode_setEnabled(self.outlightV_node, false)
		WorldNode_setEnabled(self.outlightH_node, true)
	end
	
end


function SettingsPage:chooseAltLayout(animator)
	
	self.group_current_alt[self.current_layout_group] = (self.group_current_alt[self.current_layout_group]) % VC_getLayoutGroupConfigCount(self.current_layout_group) + 1
		
	local linear = getLinearIndexFromGroupAndAlt(self.current_layout_group, self.group_current_alt[self.current_layout_group])

	local previousLayout = self.currentLayout
	self.currentLayout = linear

	self:setLayoutLabels(previousLayout, self.currentLayout)

	WorldNode_setEnabled(self.layout_node_alt, true)
	
	self:launchAnimation(animator)
	
	VC_playSound(g_menu.sound_page, 1.0)
	
end

function SettingsPage:enableBilinearOption(show, force)
	
	if not force and self.isBilinearOptionVisible == show then
		return
	end
	
	self.isBilinearOptionVisible = show
	if not show then
		local animator, stateAnimator
		if self.bilinear then
			animator = self.bilinear_on_disappear
			stateAnimator = self.bilinear_on_anim
		else
			animator = self.bilinear_off_disappear
			stateAnimator = self.bilinear_off_anim
		end
		
		-- force state before disappearing
		Component_enable(stateAnimator)
		AnimatorComponent_setCurrentTime(stateAnimator, 0.3)
		
		Component_enable(animator)
		AnimatorComponent_setSpeed(animator, 1)
		AnimatorComponent_replay(animator)
		
	else
		local animator
		if self.bilinear then
			animator = self.bilinear_on_disappear
		else
			animator = self.bilinear_off_disappear
		end
		Component_enable(animator)
		AnimatorComponent_setSpeed(animator, -1)
		AnimatorComponent_replay(animator)
	end
	
	Component_disable(self.bilinear_on_anim)
	Component_disable(self.bilinear_off_anim)
end

function SettingsPage:beginChooseLeftLayout(setLabels)
	
	self.previous_layout_group = self.current_layout_group
	self.current_layout_group = (self.current_layout_group + self.group_count - 2) % self.group_count + 1

	self.previousLayout = self.currentLayout
	self.currentLayout = getLinearIndexFromGroupAndAlt(self.current_layout_group, self.group_current_alt[self.current_layout_group])
		
	self:setLayers(false)
	
	if setLabels then
		self:setLayoutLabels(self.previousLayout, self.currentLayout)
	end
end

function SettingsPage:chooseLeftLayout()
	
	self:beginChooseLeftLayout(true)

	self:launchAnimation(self.move_right_animator)

	VC_playSound(g_menu.sound_scroll, 1.0)

end

function isBilinearDisabledForLayout(layout_index)
	return VC_isScreenConfigurationPixelPerfect(layout_index)
end

function SettingsPage:beginChooseRightLayout(setLabels)
	
	self.previous_layout_group = self.current_layout_group
	self.current_layout_group = (self.current_layout_group) % self.group_count + 1
	
	self.previousLayout = self.currentLayout
	self.currentLayout = getLinearIndexFromGroupAndAlt(self.current_layout_group, self.group_current_alt[self.current_layout_group])
	
	self:setLayers(true)
	
	if setLabels then
		self:setLayoutLabels(self.previousLayout, self.currentLayout)
	end
end

function SettingsPage:chooseRightLayout()
	
	self:beginChooseRightLayout(true)

	self:launchAnimation(self.move_left_animator)

	VC_playSound(g_menu.sound_scroll, 1.0)

end

function SettingsPage:setLayoutLabels(previousLayout, currentLayout)
	LabelComponent_setTextFromStringId(self.label_layout_name_alt, VC_getScreenConfigurationName(previousLayout))
	LabelComponent_setTextFromStringId(self.label_layout_desc_alt, VC_getScreenConfigurationDesc(previousLayout))
	LabelComponent_setTextFromStringId(self.label_layout_name, VC_getScreenConfigurationName(currentLayout))
	LabelComponent_setTextFromStringId(self.label_layout_desc, VC_getScreenConfigurationDesc(currentLayout))
end

function SettingsPage:displaySwitchHandIcon(layoutGroup)
	
	if layoutGroup == nil then
		layoutGroup = self.current_layout_group
	end
	
	WorldNode_setEnabled(self.switch_hand_node, VC_getLayoutGroupConfigCount(layoutGroup) > 1)
		
	local sheet_sprite = WorldNode_getComponentByTypeName(self.switch_hand_node, "sheet_sprite")
	
	if self.group_current_alt[layoutGroup] == 2 then
		SheetSpriteComponent_setFrameByName(sheet_sprite, "NTR_switchLeftHand.png")
	else
		SheetSpriteComponent_setFrameByName(sheet_sprite, "NTR_switchRightHand.png")
	end
end


TV_SIZEX = 378
TV_SIZEY = 212
DRC_SIZEX = 158
DRC_SIZEY = 90
TV_OFFSETX = 0
TV_OFFSETY = 10
DRC_OFFSETX = 0
DRC_OFFSETY = 0 -- -15

TV_BG_NAME = "nds1st_31p_tv.png"
DRC_BG_NAME = "nds1st_31p.png"

function computeLayoutPres(drc_node, drc_screen_bottom, drc_screen_top, tv_screen_bottom, tv_screen_top, layout_index)
	
	-- print("----")
	local drcRotation = VC_getScreenDrcRotation(layout_index)

	computeScreen(drc_screen_bottom, layout_index, DISPLAY_TARGET_DRC, DISPLAY_SOURCE_LOWER, DRC_SIZEX, DRC_SIZEY, DRC_OFFSETX, DRC_OFFSETY, DRC_REF_SIZE_X, DRC_REF_SIZE_Y)
	computeScreen(drc_screen_top,    layout_index, DISPLAY_TARGET_DRC, DISPLAY_SOURCE_UPPER, DRC_SIZEX, DRC_SIZEY, DRC_OFFSETX, DRC_OFFSETY, DRC_REF_SIZE_X, DRC_REF_SIZE_Y)
	computeScreen(tv_screen_bottom,  layout_index, DISPLAY_TARGET_TV,  DISPLAY_SOURCE_LOWER, TV_SIZEX,  TV_SIZEY,  TV_OFFSETX,  TV_OFFSETY,  TV_REF_SIZE_X,  TV_REF_SIZE_Y)
	computeScreen(tv_screen_top,     layout_index, DISPLAY_TARGET_TV,  DISPLAY_SOURCE_UPPER, TV_SIZEX,  TV_SIZEY,  TV_OFFSETX,  TV_OFFSETY,  TV_REF_SIZE_X,  TV_REF_SIZE_Y)
	
	
	-- print("drc rotation = "..drcRotation)
	
	WorldNode_setLocalRotation(drc_node, math.rad(drcRotation))

end

function string.ends(String,End)
   return End=='' or string.sub(String,-string.len(End))==End
end

function SettingsPage:computePresentationOnNode(node, linear)
	local drc_node = WorldNode_getChildByName(node, "drc_icon")
	local tv_node = WorldNode_getChildByName(node, "tv_icon")
	
	local drc_bg = WorldNode_getChildByName(drc_node, "drc_background")
	local tv_bg = WorldNode_getChildByName(tv_node, "tv_background")
	
	local tv_bg_name = VC_getScreenLayoutBackgroundName(linear, 0)
	local drc_bg_name = VC_getScreenLayoutBackgroundName(linear, 1)
	
	-- Patch to show Nintendo DS background on specific layout
	if string.sub(tv_bg_name,-string.len(TV_BG_NAME))==TV_BG_NAME then
		WorldNode_setEnabled(tv_bg, true)
	else
		WorldNode_setEnabled(tv_bg, false)
	end
	
	if string.sub(drc_bg_name,-string.len(DRC_BG_NAME))==DRC_BG_NAME then
		WorldNode_setEnabled(drc_bg, true)
		
	else
		WorldNode_setEnabled(drc_bg, false)
	end	
	
	local drc_screen_bottom = WorldNode_getChildByName(drc_node, "drc_screen_bottom")
	local drc_screen_top = WorldNode_getChildByName(drc_node, "drc_screen_top")
	local tv_screen_bottom = WorldNode_getChildByName(tv_node, "tv_screen_bottom")
	local tv_screen_top = WorldNode_getChildByName(tv_node, "tv_screen_top")
	
	computeLayoutPres(drc_node, drc_screen_bottom, drc_screen_top, tv_screen_bottom, tv_screen_top, linear)
	
end


function SettingsPage:computeLayoutsPresentation()
	
	local size = self.group_count
	
	local linear = 0
	local group = self.current_layout_group
	local first_index = 5 - self.current_layout_group - math.max(1, math.floor(self.group_count / 2))
	
	-- print("group = " .. group)
	-- print("first_index = " .. first_index)

	-- rewind 2 steps	
	group = (group + size - 3) % size + 1
	

	-- print("---")	
	for layout_node = 1, 5, 1 do
		
		local visible = self.can_cycle or (first_index <= layout_node and layout_node < (first_index + self.group_count))
		
		-- print("layout \t" .. layout_node .. "visible = " .. tostring(visible))
		
		WorldNode_setEnabled(self.layout_nodes[layout_node], visible)
		
		if visible then
			-- print(" cur alt \t".. self.group_current_alt[group] )
			
			linear = getLinearIndexFromGroupAndAlt(group, self.group_current_alt[group])
			
				
			self:computePresentationOnNode(self.layout_nodes[layout_node], linear)
	
			-- Selected layout computation
			if layout_node == 3 then
				
				--print(" linear \t"..linear)
				
				local drcRotation = VC_getScreenDrcRotation(linear)
				
				-- choose correct outlight sprite
				if drcRotation == 90 or drcRotation == 270 then
					VisualComponent_setAlpha(self.outlightV, 1.0)
					VisualComponent_setAlpha(self.outlightH, 0.0)
				else
					VisualComponent_setAlpha(self.outlightV, 0.0)
					VisualComponent_setAlpha(self.outlightH, 1.0)
				end
			
				-- compute layout for alt layout
				if VC_getLayoutGroupConfigCount(self.current_layout_group) > 1 then
					if self.group_current_alt[group] == 1 then
						alt_index = 2
					else
						alt_index = 1
					end
					linear = getLinearIndexFromGroupAndAlt(group, alt_index)
					self:computePresentationOnNode(self.layout_node_alt, linear)
				end
			end
		end

		group = (group) % size + 1

	end
	
	
	self:displaySwitchHandIcon()
	
end
