require 'scripts/gui/gui.lua'
require 'scripts/gui/gui_button.lua'


GUI.dragThresholdX = 20
GUI.dragThresholdY = math.huge


LayoutScroller = class(GUIButton)


LayoutScroller.ANIM_DURATION = 0.5
LayoutScroller.ANIM_DISTANCE = 495
LayoutScroller.DIST_TO_TIME = LayoutScroller.ANIM_DURATION / LayoutScroller.ANIM_DISTANCE * 0.8

LayoutScroller.SWITCH_TIME = 0.2
LayoutScroller.SWITCH_HYSTERESIS = 0.01
LayoutScroller.SWITCH_TIME_NEXT = LayoutScroller.SWITCH_TIME + LayoutScroller.SWITCH_HYSTERESIS
LayoutScroller.SWITCH_TIME_PREV = LayoutScroller.SWITCH_TIME - LayoutScroller.SWITCH_HYSTERESIS

LayoutScroller.SWITCH_ITEM_TIME = LayoutScroller.SWITCH_TIME * 0.5

LayoutScroller.MAX_INERTIA_SPEED = 10000
LayoutScroller.INERTIA_SPEED_THRESHOLD = 1000
LayoutScroller.INERTIA_COOLDOWN_FACTOR = 5
LayoutScroller.MIN_INERTIA_SPEED_TO_SWITCH = 3000

--------------------------------------------------------------------------------
-- sendCommand
--! @brief Function to handle a gui command sent by the GUI system to the element
--! @param command command to handle
--------------------------------------------------------------------------------
function LayoutScroller:sendCommand(command)
	
	if not Component_isEnabled(self.clickable) then
		return false
	end
	
	if command.id == GUI_COMMAND_PRESS then
		
		if self.settings_page.state ~= SETTINGS_STATE_IDLE then
			self.onlyPressed = true -- scroller is only pressed, not dragged
			
			if self.settings_page.state == SETTINGS_STATE_ANIM then
				
				self.animator = self.settings_page.animation_controller
				AnimatorComponent_stop(self.animator)
				self:setEndLayoutIfRollbacked()
				
			elseif self.settings_page.state == SETTINGS_STATE_SCROLLING then
				
				self:setEndLayoutIfRollbacked()
				
			end
			
			self:startScroll()
		end
	
	elseif command.id == GUI_COMMAND_DRAG then
		
		local localPosX = command.posX --WorldNode_worldToLocalPosition(self.worldNode, command.posX, command.posY)
		local localPrevPosX = command.previousPosX --WorldNode_worldToLocalPosition(self.worldNode, command.previousPosX, command.previousPosY)
		
		self.dx = localPosX - localPrevPosX
		
		-- start dragging
		if command.first then
			
			self.onlyPressed = false
			
			if self.settings_page.state == SETTINGS_STATE_IDLE then
				
				if self.dx < 0 then
					self.animator = self.animator_left
					self.settings_page:beginChooseRightLayout(true)
				else
					self.animator = self.animator_right
					self.settings_page:beginChooseLeftLayout(true)
				end
				
				AnimatorComponent_setSpeed(self.animator, 1)
				VC_playSound(g_menu.sound_scroll, 1.0)
				
			elseif self.settings_page.state == SETTINGS_STATE_ANIM then
			
				self.animator = self.settings_page.animation_controller
				AnimatorComponent_stop(self.animator)
				self:setEndLayoutIfRollbacked()
				
			elseif self.settings_page.state == SETTINGS_STATE_SCROLLING then
				
				self:setEndLayoutIfRollbacked()
				
			end
			
			self:startScroll()
		end
		
		local dtAnim = self.dx * LayoutScroller.DIST_TO_TIME
		self:scroll(dtAnim)

		return true
		
	elseif command.id == GUI_COMMAND_RELEASE then
		
		if self.onlyPressed then
			self.onlyPressed = false
			self:endScroll()
		elseif command.endDrag then
			self.speed = self.dx / GUI.dt
			self.speed = Math.clamp(self.speed, -LayoutScroller.MAX_INERTIA_SPEED, LayoutScroller.MAX_INERTIA_SPEED)
			self.hasInertia = true
		end
	end
	
	return GUIButton.sendCommand(self, command)
end

function LayoutScroller:setEndLayoutIfRollbacked()
	local speed = AnimatorComponent_getSpeed(self.animator)
	if speed < 0 then
		if self.animator == self.animator_right then
			self.settings_page:beginChooseLeftLayout(false)
		elseif self.animator == self.animator_left then
			self.settings_page:beginChooseRightLayout(false)
		end
		AnimatorComponent_setSpeed(self.animator, 1)
	end
end


function LayoutScroller:update(dt)
	if self.hasInertia then
		
		if math.abs(self.speed) > LayoutScroller.INERTIA_SPEED_THRESHOLD then
			self:scroll(self.speed * dt * LayoutScroller.DIST_TO_TIME)
			self.speed = Math.lerp(self.speed, 0, LayoutScroller.INERTIA_COOLDOWN_FACTOR * dt)
		else
			self.hasInertia = false
			self:endScroll()
		end
	end
end


function LayoutScroller:startScroll()
	
	self.settings_page.state = SETTINGS_STATE_SCROLLING
	self.settings_page:enableClickables(false, false, true)
	
	self.hasInertia = false
	
	Component_enable(self.settings_page.label_layout_name_alt)
	Component_enable(self.settings_page.label_layout_desc_alt)
	
	Component_disable(self.settings_page.outlightH_animator)
	Component_disable(self.settings_page.outlightV_animator)
end

function LayoutScroller:scroll(dt)
	
	if self.animator == self.animator_left then
		dt = -dt
	end
	
	local time = AnimatorComponent_getCurrentTime(self.animator)
		
	-- handle change of layout
	if time + dt < 0 then
		
		if self.hasInertia and math.abs(self.speed) < LayoutScroller.MIN_INERTIA_SPEED_TO_SWITCH then
			self.speed = 0
		else
			VC_playSound(g_menu.sound_scroll, 1.0)
			
			if self.animator == self.animator_right then
				self.animator = self.animator_left
				self.settings_page:beginChooseRightLayout(true)
				self.settings_page:beginChooseRightLayout(true)
			elseif self.animator == self.animator_left then
				self.animator = self.animator_right
				self.settings_page:beginChooseLeftLayout(true)
				self.settings_page:beginChooseLeftLayout(true)
			end
			
			AnimatorComponent_setSpeed(self.animator, 1)
			AnimatorComponent_reset(self.animator)
			
			dt = -(time + dt)
		end
	
	elseif time + dt > LayoutScroller.ANIM_DURATION then
	
		if self.hasInertia and math.abs(self.speed) < LayoutScroller.MIN_INERTIA_SPEED_TO_SWITCH then
			self.speed = 0
		else
			VC_playSound(g_menu.sound_scroll, 1.0)
		
			self.settings_page:computeLayoutsPresentation()
			
			if self.animator == self.animator_left then
				self.settings_page:beginChooseRightLayout(true)
			elseif self.animator == self.animator_right then
				self.settings_page:beginChooseLeftLayout(true)
			end
			
			AnimatorComponent_setSpeed(self.animator, 1)
			AnimatorComponent_reset(self.animator)
			
			dt = (time + dt) % LayoutScroller.ANIM_DURATION
		end
	end
	
	-- handle change of layout params (bilinear, outlight, hand icon)
	if time < LayoutScroller.SWITCH_TIME_NEXT and time + dt >= LayoutScroller.SWITCH_TIME_NEXT then
		
		local isBilinearEnabled  = not isBilinearDisabledForLayout(self.settings_page.currentLayout)
		self.settings_page:enableBilinearOption(isBilinearEnabled)
		
		self.settings_page:selectOutlight(self.settings_page.currentLayout)
		
		self.settings_page:displaySwitchHandIcon(self.settings_page.current_layout_group)
		
	elseif time > LayoutScroller.SWITCH_TIME_PREV and time + dt <= LayoutScroller.SWITCH_TIME_PREV then
		
		local wasBilinearEnabled = not isBilinearDisabledForLayout(self.settings_page.previousLayout)
		self.settings_page:enableBilinearOption(wasBilinearEnabled)
		
		self.settings_page:selectOutlight(self.settings_page.previousLayout)
		
		self.settings_page:displaySwitchHandIcon(self.settings_page.previous_layout_group)
	end
	
	
	time = AnimatorComponent_getCurrentTime(self.animator)
	AnimatorComponent_setCurrentTime(self.animator, time + dt)
end


function LayoutScroller:endScroll()
	
	local time = AnimatorComponent_getCurrentTime(self.animator)
	
	if (time > LayoutScroller.SWITCH_ITEM_TIME) then
		
		-- finish to next
		AnimatorComponent_setSpeed(self.animator, 1)
		
	else
		
		-- roll back to previous
		AnimatorComponent_setSpeed(self.animator, -1)
		
		if self.animator == self.animator_left then
			self.settings_page:beginChooseLeftLayout(false)
		else
			self.settings_page:beginChooseRightLayout(false)
		end
	end
	
	self.settings_page:launchAnimation(self.animator, true)
end

