require 'scripts/gui/gui.lua'

--------------------------------------------------------------------------------
-- GUIPopup Object script
--! @class GUIPopup
--! 
--! Attributes
--! @variable {Component} [listener] script component that listens to the popup events.
--! @variable {Component} [blockingClickable] clickable area blocking access to below elements. If not defined it is automatically created. Its size is GUI.cameraViewportWidth x GUI.cameraViewportHeight
--! @variable {Number}    [layer] minimum layer of popup. Every clickable objects below this layer won't be accessible. Default 0
--! @variable {Component} [ok_button] reference to the ok button
--! @variable {Component} [cancel_button] reference to the cancel button
--! @variable {Component} [yes_button] reference to the yes button
--! @variable {Component} [no_button] reference to the no button
--! 
--! Events
--! * onPopupClose(popup, userChoice)
--!
--------------------------------------------------------------------------------

GUIPopup = class()

--------------------------------------------------------------------------------
-- start
--! @brief Callback when the object is added to the world
--------------------------------------------------------------------------------
function GUIPopup:start()
	GUI:debugPrint("[" .. tostring(self) .. "]\tGUIPopup:start()", "runtime")
	if not self.layer then
		self.layer = 0
	end
	
	if self.ok_button then
		self.ok_button = ScriptComponent_getScriptTable(self.ok_button)
	end
	
	if self.cancel_button then
		self.cancel_button = ScriptComponent_getScriptTable(self.cancel_button)
	end
	
	if self.yes_button then
		self.yes_button = ScriptComponent_getScriptTable(self.yes_button)
	end
	
	if self.no_button then
		self.no_button = ScriptComponent_getScriptTable(self.no_button)
	end
	
	if self.blockingClickable == nil then
		GUI:debugPrint("[" .. tostring(self) .. "]\tGUIPopup create blocking area", "hit_test")
		self.blockingClickable = addNewComponentToNode(self.worldNode, COMPONENT_TYPE_CLICKABLE)
		ClickableComponent_setLayer(self.blockingClickable, self.layer)
		ClickableComponent_setBoxShape(self.blockingClickable, GUI.cameraViewportWidth, GUI.cameraViewportHeight)
	end
	
	GUI:focusElement(nil)
end

--------------------------------------------------------------------------------
-- stop
--! @brief Callback when object is removed from the world
--------------------------------------------------------------------------------
function GUIPopup:stop()
	GUI:debugPrint("[" .. tostring(self) .. "]\tGUIPopup:stop()", "runtime")
	
	if self.blockingClickable ~= nil then
		removeAndDestroyComponentFromNode(self.worldNode, self.blockingClickable)
	end
	
end

--------------------------------------------------------------------------------
-- close
--! @brief Closes the popup using the given choice and call listener callback
--! @param choice
--------------------------------------------------------------------------------
function GUIPopup:close(choice)
	if self.listener ~= nil then
		self.listener:onPopupClose(self, choice)
	end
	
	WorldNode_setEnabled(self.worldNode, false)
end

--------------------------------------------------------------------------------
-- onButtonClick
--! @brief Callback when a button is clicked
--! @param button Button that has been clicked
--------------------------------------------------------------------------------
function GUIPopup:onButtonClick(button)
	GUI:debugPrint("[" .. tostring(self) .. "]\tGUIPopup:onButtonClick()", "callbacks")
	if button == self.ok_button then
		self:close("ok")
	elseif button == self.cancel_button then
		self:close("cancel")
	elseif button == self.yes_button then
		self:close("yes")
	elseif button == self.no_button then
		self:close("no")
	end
	
end

--------------------------------------------------------------------------------
-- disabled
--! @brief Disables the entire popup by set all buttons in disabled state
--------------------------------------------------------------------------------
function GUIPopup:disable()
	if self.ok_button then
		self.ok_button:setState("disabled")
	end
	if self.cancel_button then
		self.cancel_button:setState("disabled")
	end
	if self.yes_button then
		self.yes_button:setState("disabled")
	end
	if self.no_button then
		self.no_button:setState("disabled")
	end
end

--------------------------------------------------------------------------------
-- enable
--! @brief Enables the entire popup by set all buttons in idle state
--------------------------------------------------------------------------------
function GUIPopup:enable()
	if self.ok_button then
		self.ok_button:setState("idle")
	end
	if self.cancel_button then
		self.cancel_button:setState("idle")
	end
	if self.yes_button then
		self.yes_button:setState("idle")
	end
	if self.no_button then
		self.no_button:setState("idle")
	end
end

