require 'scripts/gui/gui_button.lua'

--------------------------------------------------------------------------------
-- GUICheckButton Object script
--! @class GUICheckButton
--! 
--! States
--! * idle
--! * disabled
--! * pressed
--!
--! Attributes
--! @variable {Component} [listener] script component that listens to the element events.
--! @variable {Component} [clickable] area for click detection. If not defined, it is automatically created.
--! @variable {Boolean}   [autoAdjustClickableArea] flag that determines automatic adjusment of the clickable area according to the currrent visual. Default true when no clickable is defined else false.
--! @variable {Component} [idleVisual] visual for idle state.
--! @variable {Component} [focusedVisual] visual for focused state.
--! @variable {Component} [pressedVisual] visual for pressed state.
--! @variable {Component} [checkedVisual] visual for checked state
--! @variable {Component} [uncheckedVisual] visual for unchecked state
--! @variable {Component} [checkedPressVisual] visual for checked state in pressed mode. If not provided, it fallbacks on checkVisual
--! @variable {Component} [uncheckedPressVisual] visual for unchecked state. If not provided, it fallbacks on idleVisual
--! @variable {Boolean}   [checked] initial check state. Default false
--! @variable {Boolean}   [autoFocusDisabled] cannot be auto-focused through key navigation. Default false

--!
--! Events
--! * onButtonPress(button, positionX, positionY)
--! * onButtonRelease(button, positionX, positionY)
--! * onButtonClick(button)
--! * onCheckToggle(check, checked_flag)
--!
--------------------------------------------------------------------------------

GUICheckButton = class(GUIButton)

GUICheckButton.states = class(GUIButton.states)

--------------------------------------------------------------------------------
-- State idle callbacks
--------------------------------------------------------------------------------
GUICheckButton.states.idle = class(GUIButton.states.idle) 
GUICheckButton.states.idle.enter = function(checkButton, previousState)
	GUI:debugPrint("GUICheckButton : enter idle " .. tostring(checkButton), "states")
		
	if not checkButton.checked then
		checkButton:disableVisual(checkButton.checkedVisual)
		checkButton:enableVisual(checkButton.uncheckedVisual)
	else
		checkButton:disableVisual(checkButton.uncheckedVisual)
		checkButton:enableVisual(checkButton.checkedVisual)
	end
end

GUICheckButton.states.idle.leave = function(checkButton, nextState)
	GUI:debugPrint("GUICheckButton : leave idle " .. tostring(checkButton), "states")

	checkButton:disableVisual(checkButton.checkedVisual)
	checkButton:disableVisual(checkButton.uncheckedVisual)

end


--------------------------------------------------------------------------------
-- State pressed callbacks
--------------------------------------------------------------------------------
GUICheckButton.states.pressed = class(GUIButton.states.pressed)
GUICheckButton.states.pressed.enter = function(checkButton, previousState)
	GUI:debugPrint("GUICheckButton : enter pressed " .. tostring(checkButton), "states")
		
	if not checkButton.checked then
		checkButton:disableVisual(checkButton.checkedPressVisual)
		checkButton:enableVisual(checkButton.uncheckedPressVisual)
	else
		checkButton:disableVisual(checkButton.uncheckedPressVisual)
		checkButton:enableVisual(checkButton.checkedPressVisual)
	end
	
end
	
GUICheckButton.states.pressed.leave = function(checkButton, nextState)
	GUI:debugPrint("GUICheckButton : leave pressed " .. tostring(checkButton), "states")
	
	checkButton:disableVisual(checkButton.checkedPressVisual)
	checkButton:disableVisual(checkButton.uncheckedPressVisual)

end
	
--------------------------------------------------------------------------------
-- start
--! @brief Callback when the object is added to the world
--------------------------------------------------------------------------------
function GUICheckButton:start()
	GUI:debugPrint("[" .. tostring(self) .. "]\tGUICheckButton:start()", "runtime")

	if not self.uncheckedVisual then
		self.uncheckedVisual = self.idleVisual
	end

	if not self.uncheckedPressVisual then
		self.uncheckedPressVisual = self.pressedVisual
	end
	if not self.uncheckedPressVisual then
		self.uncheckedPressVisual = self.idleVisual
	end
	
	if not self.checkedPressVisual then
		self.checkedPressVisual = self.checkedVisual
	end

	self:disableVisual(self.checkedPressVisual)
	self:disableVisual(self.uncheckedPressVisual)
	self:disableVisual(self.checkedVisual)
	self:disableVisual(self.uncheckedVisual)
	
	GUIButton.start(self)

	if self.checked == nil then
		self:setChecked(false)
	else
		self:setChecked(self.checked)
	end
	
end


--------------------------------------------------------------------------------
-- sendCommand
--! @brief Function to handle a gui command sent by the GUI system to the element
--! @param command command to handle
--------------------------------------------------------------------------------
function GUICheckButton:sendCommand(command)
	GUI:debugPrint("[" .. tostring(self) .. "]\tGUICheckButton:sendCommand() " .. tostring(command.id), "commands")
		
	if command.id == GUI_COMMAND_CLICK then
		GUI:debugPrint("[" .. tostring(self) .. "]\tGUICheckButton clicked ", "actions")
		
		self:setChecked(not self.checked)
		
		if self.listener ~= nil and self.listener.onCheckToggle ~= nil then
			self.listener:onCheckToggle(self, self.checked)
		end

		return true
		
	else 
		return GUIButton.sendCommand(self, command)
	end
	
end

--------------------------------------------------------------------------------
-- setChecked
--! @brief Sets the checked flag and change the visual accordingly
--! @param checked new flag
--------------------------------------------------------------------------------
function GUICheckButton:setChecked(checked)
	
	if self.checked == checked then
		return
	end
	
	self.checked = checked
	
	if not self.checked then
		self:disableVisual(self.checkedVisual)
		self:enableVisual(self.uncheckedVisual)
	else
		self:disableVisual(self.uncheckedVisual)
		self:enableVisual(self.checkedVisual)
	end
		
end

--------------------------------------------------------------------------------
-- isChecked
--! @brief Returns whether the button is checked or not
--------------------------------------------------------------------------------
function GUICheckButton:isChecked()
	return self.checked
end

