require 'scripts/gui/gui_element.lua'

--------------------------------------------------------------------------------
-- GUIButton Object script
--! @class GUIButton
--! @parent GUIElement
--! 
--! States
--! * idle
--! * disabled
--! * pressed
--!
--! Attributes 
--! @variable {Component} [listener] script component that listens to the element events.
--! @variable {Component} [clickable] area for click detection. If not defined, it is automatically created.
--! @variable {Boolean}   [autoAdjustClickableArea] flag that determines automatic adjusment of the clickable area according to the currrent visual. Default true when no clickable is defined else false.
--! @variable {Component} [idleVisual] visual for idle state.
--! @variable {Component} [focusedVisual] visual for focused state.
--! @variable {Component} [pressedVisual] visual for pressed state.
--! @variable {Component} [pressedAnimation] reference to the animator to launch on press
--! @variable {Component} [releasedAnimation] reference to the animator to launch on release
--! @variable {Component} [clickedAnimation] reference to the animator to launch on click
--! @variable {Component} [pressedSound] reference to the sound component to launch on press
--! @variable {Component} [releasedSound] reference to the sound component to launch on release
--! @variable {Component} [clickedSound] reference to the sound component to launch on click
--! @variable {Boolean}   [autoFocusDisabled] : cannot be auto-focused through key navigation
--!
--! Events
--! * onElementFocus(element)
--! * onElementUnfocus(element)
--! * onButtonPress(button, positionX, positionY)
--! * onButtonRelease(button, positionX, positionY)
--! * onButtonClick(button)
--!
--------------------------------------------------------------------------------

GUIButton = class(GUIElement)

GUIButton.states = class(GUIElement.states)

--------------------------------------------------------------------------------
-- State pressed callbacks
--------------------------------------------------------------------------------
GUIButton.states.pressed = class()

GUIButton.states.pressed.enter = function(element, previousState)
	GUI:debugPrint("GUIButton : enter pressed " .. tostring(element), "states")
	if element.pressedVisual ~= nil then
		element:enableVisual(element.pressedVisual)
	else
		element:enableVisual(element.idleVisual)
	end
	
	if element.pressedAnimation ~= nil then
		GUI:debugPrint("GUIButton : play pressed animation", "animations")
		Component_enable(element.pressedAnimation)
		AnimatorComponent_setSpeed(element.pressedAnimation, 1.0)
		AnimatorComponent_play(element.pressedAnimation)
	end
	
	if element.pressedSound ~= nil then
		GUI:debugPrint("GUIButton : play pressed sound", "sounds")
		SoundComponent_stop(element.pressedSound)
		SoundComponent_play(element.pressedSound)
	end
end
	
GUIButton.states.pressed.leave = function(element, nextState)
	GUI:debugPrint("GUIButton : leave pressed " .. tostring(element), "states")
	if element.pressedVisual ~= nil then
		element:disableVisual(element.pressedVisual)
	else
		element:disableVisual(element.idleVisual)
	end
	
	if element.pressedAnimation ~= nil then --and (not element.clickedAnimation or not Component_isEnabled(element.clickedAnimation) or not AnimatorComponent_isPlaying(element.clickedAnimation)) then
		GUI:debugPrint("GUIButton : play released animation", "animations")
		--Component_enable(element.releasedAnimation)
		--AnimatorComponent_play(element.releasedAnimation)
		AnimatorComponent_setSpeed(element.pressedAnimation, -1.0)
		AnimatorComponent_play(element.pressedAnimation)
	end
	
	if element.releasedSound ~= nil then
		GUI:debugPrint("GUIButton : play released sound", "sounds")
		SoundComponent_stop(element.releasedSound)
		SoundComponent_play(element.releasedSound)
	end
end



--------------------------------------------------------------------------------
-- start
--! @brief Callback when the object is added to the world
--------------------------------------------------------------------------------
function GUIButton:start()

	GUI:debugPrint("[" .. tostring(self) .. "]\tGUIButton:start()", "runtime")
	self:setupClickableArea()
	self:disableVisual(self.pressedVisual)
	
	self:setupAnimator(self.clickedAnimation)
	self:setupAnimator(self.pressedAnimation)
	self:setupAnimator(self.releasedAnimation)
	
	GUIElement.start(self)
	
	GUI:registerButton(self)
	
end

--------------------------------------------------------------------------------
-- stop
--! @brief Callback when the object is removed from to the world
--------------------------------------------------------------------------------
function GUIButton:stop()
	
	GUI:unregisterButton(self)
	
	GUIElement.stop(self)
end

--------------------------------------------------------------------------------
-- playClickedEffects
--! @brief Plays the clicked animation and clicked sound of this button, if any
--------------------------------------------------------------------------------
function GUIButton:playClickedEffects()
	self:playClickedAnimation()
	self:playClickedSound()
end

--------------------------------------------------------------------------------
-- playClickedAnimation
--! @brief Plays the clicked animation referenced by the clickedAnimation attribute
--------------------------------------------------------------------------------
function GUIButton:playClickedAnimation()
	if self.clickedAnimation ~= nil then
		GUI:debugPrint("GUIButton : play clicked animation", "animations")
		self:playAnimation(self.clickedAnimation)
	end
end

--------------------------------------------------------------------------------
-- playClickedSound
--! @brief Plays the clicked sound referenced by the clickedSound attribute
--------------------------------------------------------------------------------
function GUIButton:playClickedSound()
	if self.clickedSound then
		GUI:debugPrint("GUIButton : play clicked sound", "sounds")
		SoundComponent_stop(self.clickedSound)
		SoundComponent_play(self.clickedSound)
	end
end

--------------------------------------------------------------------------------
-- sendCommand
--! @brief Function to handle a gui command sent by the GUI system to the element
--! @param command command to handle
--------------------------------------------------------------------------------
function GUIButton:sendCommand(command)
	GUI:debugPrint("[" .. tostring(self) .. "]\tGUIButton:sendCommand() " .. tostring(command.id), "commands")
		
	if command.id == GUI_COMMAND_PRESS then
		GUI:debugPrint("[" .. tostring(self) .. "]\tGUIButton pressed ", "actions")
		self:setState("pressed")
		if self.listener ~= nil and self.listener.onButtonPress ~= nil then
			self.listener:onButtonPress(self, command.posX, command.posY)
		end

		return true
		
	elseif command.id == GUI_COMMAND_RELEASE then
		GUI:debugPrint("[" .. tostring(self) .. "]\tGUIButton released ", "actions")
		self:setState("idle")
		if self.listener ~= nil and self.listener.onButtonRelease ~= nil then
			self.listener:onButtonRelease(self, command.posX, command.posY)
		end

		
		return true
		
	elseif command.id == GUI_COMMAND_CLICK then
		GUI:debugPrint("[" .. tostring(self) .. "]\tGUIButton clicked ", "actions")
		
		if self.listener ~= nil and self.listener.onButtonClick ~= nil then
			self.listener:onButtonClick(self)
		end

		self:playClickedEffects()
		
		return true
	else 
		return GUIElement.sendCommand(self, command)
	end
	
end
